// ==========================================================================
//
// = LIBRARY
//     PCIEFPGA
// 
// = FILENAME
//     pciefpgacspkt.cpp
// 
// = COPYRIGHT
//     (C) Copyright 2007 Agilent Technologies
//
// ==========================================================================
#include "PCIEFPGACSPkt.h"
#include "devpcieintdefs.h"
#include "AgtPCIEControl.h"
#include "PCIETLPInfo.h"

#define PCIEFPGATXMISC_POINTERS_FIRST_ADDR_BITPOS ((UInt8)  0)
#define PCIEFPGATXMISC_POINTERS_LAST_ADDR_BITPOS  ((UInt8) 16)
#define PCIEFPGATXMISC_POINTERS_BITSIZE           ((UInt8)  9)

//***************************************************************************
CPCIEFpgaCSPkt::CPCIEFpgaCSPkt()
{
  mTxCtrl                 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_TXCTRL;
  mTxCtrlExtended         = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_TXCTRL_EXTENDED;
  mTxPktStatus            = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_TX_PKT_STATUS;
  mTxPktStatusHwChannels  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_TXN_STATUS_HW_CHANNELS;
  mTxPktStatusExtended    = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_TX_PKT_STATUS_EXTENDED;

  mBlockAPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_A_POINTERS;
  mBlockBPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_B_POINTERS;
  mBlockCPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_C_POINTERS;
  mBlockDPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_D_POINTERS;
  mBlockEPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_E_POINTERS;
  mBlockBVF1Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_BVF1_POINTERS;
  mBlockBVF2Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_BVF2_POINTERS;
  mBlockCVF1Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_CVF1_POINTERS;
  mBlockCVF2Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_CVF2_POINTERS;
  mBlockDVF1Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_DVF1_POINTERS;
  mBlockDVF2Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_DVF2_POINTERS;
  mBlockEVF1Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_EVF1_POINTERS;
  mBlockEVF2Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_EVF2_POINTERS;

  mReqBehAPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_A_POINTERS;
  mReqBehBPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_B_POINTERS;
  mReqBehCPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_C_POINTERS;
  mReqBehDPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_D_POINTERS;
  mReqBehEPointers     = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_E_POINTERS;
  mReqBehBVF1Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_BVF1_POINTERS;
  mReqBehBVF2Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_BVF2_POINTERS;
  mReqBehCVF1Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_CVF1_POINTERS;
  mReqBehCVF2Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_CVF2_POINTERS;
  mReqBehDVF1Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_DVF1_POINTERS;
  mReqBehDVF2Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_DVF2_POINTERS;
  mReqBehEVF1Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_EVF1_POINTERS;
  mReqBehEVF2Pointers  = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_EVF2_POINTERS;

  mReqComplQueue0Pointers = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQCOMPL_QUEUE0_POINTERS;
  mReqComplQueue1Pointers = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQCOMPL_QUEUE1_POINTERS;
  mReqComplQueue2Pointers = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQCOMPL_QUEUE2_POINTERS;
  mReqComplQueue3Pointers = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQCOMPL_QUEUE3_POINTERS;
  mReqComplQueue4Pointers = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQCOMPL_QUEUE4_POINTERS;

  mBlockRepeatFirst = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_REPEAT_FIRST;
  mBlockRepeatSecond = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_REPEAT_SECOND;
  mBlockRepeatThird = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_REPEAT_THIRD;
  mBlockRepeatFourth = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_REPEAT_FOURTH;
  mBlockRepeatFifth = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_BLOCK_REPEAT_FIFTH;

  mReqBehRepeatFirst = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_REPEAT_FIRST;
  mReqBehRepeatSecond = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_REPEAT_SECOND;
  mReqBehRepeatThird = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_REPEAT_THIRD;
  mReqBehRepeatFourth = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_REPEAT_FOURTH;
  mReqBehRepeatFifth = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_REQBEH_REPEAT_FIFTH;

  mComplBehRepeatFirst = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_COMPLBEH_REPEAT_FIRST;
  mComplBehRepeatSecond = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_COMPLBEH_REPEAT_SECOND;

  mComplDelayQueue0 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_COMPLDELAY_QUEUE0;
  mComplDelayQueue1 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_COMPLDELAY_QUEUE1;
  mComplDelayQueue2 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_COMPLDELAY_QUEUE2;
  mComplDelayQueue3 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_COMPLDELAY_QUEUE3;
  mComplDelayQueue3 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_COMPLDELAY_QUEUE4;

  mSiCtrl = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_CTRL;
  mSiIntAddr = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_INT_ADDR;
  mSiHeader0 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_HEADER0;
  mSiHeader1 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_HEADER1;
  mSiHeader2 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_HEADER2;
  mSiHeader3 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_HEADER3;
  mSiDataDW0 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_DATADW0;
  mSiDataDW1 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_DATADW1;
  mSiDataDW2 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_DATADW2;
  mSiDataDW3 = PCIE_FPGA_CSPKT_OFFSET + PCIE_FPGAOFFSET_CSPKT_SI_DATADW3;

  for( int i = 0; i < PCIE_MAXIMUM_NUMBER_OF_PROBE_BOARDS_COPY4 ; i++ )
  {
    for( int j = 0; j < NOFHWCHANNELS; j++ )
      mSendImmediateEnabled[i][j] = 0x0;
  }

  myController = new CAgtPCIEControl(); // returns only pointer if instance exists
}

//***************************************************************************
CPCIEFpgaCSPkt::~CPCIEFpgaCSPkt()
{
  delete myController;
  myController = NULL;
}

UInt32 CPCIEFpgaCSPkt::getTxCtrl( const AgtPortHandleT& portHandle )
{
  UInt32 val;

  myController->FPGARead( portHandle, mTxCtrl, val );

  return val;
}

void CPCIEFpgaCSPkt::setTxCtrl( const AgtPortHandleT& portHandle, const UInt32& val )
{
  myController->FPGAWrite( portHandle, mTxCtrl, val );
}

UInt32 CPCIEFpgaCSPkt::getTxCtrlExtended( const AgtPortHandleT& portHandle )
{
  UInt32 val;

  myController->FPGARead( portHandle, mTxCtrlExtended, val );

  return val;
}

void CPCIEFpgaCSPkt::setTxCtrlExtended( const AgtPortHandleT& portHandle, const UInt32& val )
{
  myController->FPGAWrite( portHandle, mTxCtrlExtended, val );
}

UInt32 CPCIEFpgaCSPkt::getTxPktStatus( const AgtPortHandleT& portHandle )
{
  UInt32 val;

  myController->FPGARead( portHandle, mTxPktStatus, val );

  return val;
}

void CPCIEFpgaCSPkt::setTxPktStatus( const AgtPortHandleT& portHandle, const UInt32& val )
{
  myController->FPGAWrite( portHandle, mTxPktStatus, val );
}

UInt32 CPCIEFpgaCSPkt::getTxPktStatusHwChannels( const AgtPortHandleT& portHandle )
{
  UInt32 val;

  myController->FPGARead( portHandle, mTxPktStatusHwChannels, val );

  return val;
}

void CPCIEFpgaCSPkt::setTxPktStatusHwChannels( const AgtPortHandleT& portHandle, const UInt32& val )
{
  myController->FPGAWrite( portHandle, mTxPktStatusHwChannels, val );
}

UInt32 CPCIEFpgaCSPkt::getTxPktStatusExtendedHwChannels( const AgtPortHandleT& portHandle )
{
  UInt32 val;

  myController->FPGARead( portHandle, mTxPktStatusExtended, val );

  return val;
}

void CPCIEFpgaCSPkt::setTxPktStatusExtendedHwChannels( const AgtPortHandleT& portHandle, const UInt32& val )
{
  myController->FPGAWrite( portHandle, mTxPktStatusExtended, val );
}

UInt32 CPCIEFpgaCSPkt::getSiCtrl( const AgtPortHandleT& portHandle, const SiCtrl& ctrl )
{
  UInt32 value;

  myController->FPGARead( portHandle, mSiCtrl, value );

  switch( ctrl )
  {
  case SICTRL_SEND:
    UInt8 bitPosition;

    // Find the bit for send immediately in fpga
    switch( getValueSendImmediateArray( portHandle ) )
    {
    case PCIE_HWCHANNEL_FUNCTION_A:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELA;
      break;
    case PCIE_HWCHANNEL_FUNCTION_B:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELB;
      break;
    case PCIE_HWCHANNEL_FUNCTION_C:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELC;
      break;
    case PCIE_HWCHANNEL_FUNCTION_D:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELD;
      break;
    case PCIE_HWCHANNEL_FUNCTION_E:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELE;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF1:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELBVF1;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF2:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELBVF2;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF1:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELCVF1;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF2:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELCVF2;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF1:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELDVF1;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF2:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELDVF2;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF1:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELEVF1;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF2:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELEVF2;
      break;
    case PCIE_HWCHANNEL_FUNCTION_END:
      return 0;
    }
    return getBits( value, bitPosition, 1 );

  case SICTRL_SWTAG:
    return getBits( value, FPGATXMISC_SICTRL_SWTAG_BIT, 1 );

  case SICTRL_RESOURCE:
    if( getBits( value, FPGATXMISC_SICTRL_NOT_SRC_SIMEM_BIT, 1 ) )
    {
      if( getBits( value, FPGATXMISC_SICTRL_DATA_INDICATOR_BIT, 1 ) )
      {
        return PCIE_SIV_RESOURCE_GENERATOR;
      } 
      else
      {
        return PCIE_SIV_RESOURCE_DATAMEM;
      }
    }
    else
    {
      return PCIE_SIV_RESOURCE_SIMEM;
    }

  case SICTRL_BUSERROR:
    return getBits( value, FPGATXMISC_SICTRL_BUSERROR_START_BIT, FPGATXMISC_SICTRL_BUSERROR_BIT_SIZE );

  default:
   AGT_THROW( "Invalid SiCtrl index " );
  }
}

void CPCIEFpgaCSPkt::setSiCtrl( const AgtPortHandleT& portHandle, const SiCtrl& ctrl, const UInt32& val )
{
  UInt32 theVal;

  myController->FPGARead( portHandle, mSiCtrl, theVal );

  switch( ctrl )
  {
  case SICTRL_SEND:
    UInt8 bitPosition;

    // Enable the bit for send immediately in fpga
    switch( getValueSendImmediateArray(portHandle) )
    {
    case PCIE_HWCHANNEL_FUNCTION_A:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELA;
      break;
    case PCIE_HWCHANNEL_FUNCTION_B:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELB;
      break;
    case PCIE_HWCHANNEL_FUNCTION_C:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELC;
      break;
    case PCIE_HWCHANNEL_FUNCTION_D:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELD;
      break;
    case PCIE_HWCHANNEL_FUNCTION_E:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELE;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF1:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELBVF1;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF2:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELBVF2;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF1:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELCVF1;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF2:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELCVF2;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF1:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELDVF1;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF2:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELDVF2;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF1:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELEVF1;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF2:
      bitPosition = FPGATXMISC_SICTRL_SEND_BIT_HWCHANNELEVF2;
      break;
    case PCIE_HWCHANNEL_FUNCTION_END:
      AGT_THROW( "Invalid HW Channel" );
      break;
    }
    setBits( theVal, bitPosition, 1, val );
    break;

  case SICTRL_SWTAG:
    setBits( theVal, FPGATXMISC_SICTRL_SWTAG_BIT, 1, val );
    break;

  case SICTRL_RESOURCE:
    switch( val )
    {
    case PCIE_SIV_RESOURCE_GENERATOR:
      setBits( theVal, FPGATXMISC_SICTRL_NOT_SRC_SIMEM_BIT, 1, 1 );
      setBits( theVal, FPGATXMISC_SICTRL_DATA_INDICATOR_BIT, 1, 1 );
      break;
    case PCIE_SIV_RESOURCE_DATAMEM:
      setBits( theVal, FPGATXMISC_SICTRL_NOT_SRC_SIMEM_BIT, 1, 1 );
      setBits( theVal, FPGATXMISC_SICTRL_DATA_INDICATOR_BIT, 1, 0 );
      break;
    default:
      setBits( theVal, FPGATXMISC_SICTRL_NOT_SRC_SIMEM_BIT, 1, 0 );
      setBits( theVal, FPGATXMISC_SICTRL_DATA_INDICATOR_BIT, 1, 0 );
    }
    break;

  case SICTRL_BUSERROR:
    setBits( theVal, FPGATXMISC_SICTRL_BUSERROR_START_BIT, FPGATXMISC_SICTRL_BUSERROR_BIT_SIZE, val );
    break;

  default:
   AGT_THROW( "Invalid SiCtrl index " );
  }
  
  myController->FPGAWrite( portHandle, mSiCtrl, theVal );
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getSiHeader(const AgtPortHandleT& portHandle,const UInt32& idx)
{
  UInt32 theVal;
  switch(idx)
  {
  case 0:
    myController->FPGARead(portHandle,mSiHeader0,theVal); 
    return theVal;
  case 1:
    myController->FPGARead(portHandle,mSiHeader1,theVal); 
    return theVal;
  case 2:
    myController->FPGARead(portHandle,mSiHeader2,theVal); 
    return theVal;
  case 3:
    myController->FPGARead(portHandle,mSiHeader3,theVal); 
    return theVal;
  default:
    AGT_THROW("Invalid SiHeader index");
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setSiHeader(const AgtPortHandleT& portHandle,const UInt32& idx, const UInt32& val)
{
  switch(idx)
  {
  case 0:
    myController->FPGAWrite(portHandle,mSiHeader0,val);
    break;
  case 1:
    myController->FPGAWrite(portHandle,mSiHeader1,val);
    break;
  case 2:
    myController->FPGAWrite(portHandle,mSiHeader2,val);
    break;
  case 3:
    myController->FPGAWrite(portHandle,mSiHeader3,val);
    break;
  default:
    AGT_THROW("Invalid SiHeader index");
  }

}
// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setSiHeaderProp(const AgtPortHandleT& portHandle,const EPCIESi& prop, const UInt32& val)
{
  UInt8 hIdx = CPCIETlpInfo::getDWNum(prop);
  
  UInt32 hVal = getSiHeader(portHandle, hIdx);

  CPCIETlpInfo::setPropVal(prop,hVal,val);

  setSiHeader(portHandle, hIdx,hVal);

}
// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getSiHeaderProp(const AgtPortHandleT& portHandle,const EPCIESi& prop)
{
  UInt8 hIdx = CPCIETlpInfo::getDWNum(prop);
  
  UInt32 hVal = getSiHeader(portHandle, hIdx);

  return CPCIETlpInfo::getPropVal(prop,hVal);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setSiHeaderPropDefault(const AgtPortHandleT& portHandle,const EPCIESi& prop)
{
  UInt8 hIdx = CPCIETlpInfo::getDWNum(prop);

  UInt32 defaulVal = getSiHeaderPropDefault(prop);

  UInt32 hVal = getSiHeader(portHandle, hIdx);

  CPCIETlpInfo::setPropVal(prop,hVal,defaulVal);

  setSiHeader(portHandle, hIdx,hVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getSiHeaderPropDefault(const EPCIESi& prop)
{  
  return CPCIETlpInfo::getDefault(prop);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getSiDataDW(const AgtPortHandleT& portHandle,const UInt8& idx)
{
  UInt32 address = mSiDataDW0; 
  if(idx < FPGATXMISC_SIDATA_DWNUM)
  {
    UInt32 theVal;
    myController->FPGARead(portHandle,address + (idx * 4) ,theVal);
    return theVal;
  }
  else
  {
    AGT_THROW("Invalid SiData index ");
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setSiDataDW(const AgtPortHandleT& portHandle,const UInt8& idx, const UInt32& val)
{
  UInt32 address = mSiDataDW0; 
  if(idx < FPGATXMISC_SIDATA_DWNUM)
  {
    myController->FPGAWrite(portHandle,address + (idx * 4),val);
  }
  else
  {
    AGT_THROW("Invalid SiData index " );
  }
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getSiIntAddr(const AgtPortHandleT& portHandle)
{
  UInt32 val;
  myController->FPGARead(portHandle,mSiIntAddr,val);
  return val;
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setSiIntAddr(const AgtPortHandleT& portHandle,const UInt32& val)
{
  myController->FPGAWrite(portHandle,mSiIntAddr,val);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getStatusPropVal(const AgtPortHandleT& portHandle,const EPCIETxStatus& prop)
{
  /* get the txn pkt status for h/w channel A,B */
  UInt32 status = getTxPktStatus(portHandle);

  /* get the txn pkt status for h/w channel C,BVF1,BVF2,CVF1,CVF2 
  This is done beacuse of different register maps*/
  UInt32 statushwchannels = getTxPktStatusHwChannels( portHandle );

  /* get the txn pkt status for h/w channel D,E,DVF1,DVF2,EVF1,EVF2 
  This is done beacuse of different register maps*/
  UInt32 statushwchannelext = getTxPktStatusExtendedHwChannels( portHandle );
  
  switch(prop)
  {
  case PCIE_TXSTATUS_DBI_LEFT_RIGHT_DETECT: 
    return 0x0;// Not available for Gen2.
  case PCIE_TXSTATUS_CLOCKS_READY:
    return  0x0;// Not available for Gen2.
  case PCIE_TXSTATUS_REQHEADER_GENA_RUNNING:
    return getBits(status,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_A_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENA_IDLE:
    return getBits(status,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_A_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENB_RUNNING:
    return getBits(status,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_B_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENB_IDLE:
    return getBits(status,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_B_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENC_RUNNING:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_C_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENC_IDLE:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_C_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENBVF1_RUNNING:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_BVF1_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENBVF1_IDLE:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_BVF1_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENBVF2_RUNNING:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_BVF2_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENBVF2_IDLE:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_BVF2_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENCVF1_RUNNING:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_CVF1_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENCVF1_IDLE:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_CVF1_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENCVF2_RUNNING:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_CVF2_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENCVF2_IDLE:
    return getBits(statushwchannels,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_CVF2_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GEND_RUNNING:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_D_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GEND_IDLE:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_D_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENE_RUNNING:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_E_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENE_IDLE:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_E_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENDVF1_RUNNING:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_DVF1_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENDVF1_IDLE:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_DVF1_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENDVF2_RUNNING:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_DVF2_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENDVF2_IDLE:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_DVF2_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENEVF1_RUNNING:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_EVF1_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENEVF1_IDLE:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_EVF1_IDLE_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENEVF2_RUNNING:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_EVF2_RUNNING_BIT,1);
  case PCIE_TXSTATUS_REQHEADER_GENEVF2_IDLE:
    return getBits(statushwchannelext,FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_EVF2_IDLE_BIT,1);
  default:
    AGT_THROW("CPCIEFpgaCSPkt::getStatusPropVal: invalid prop val ");
  }
}

// -----------------------------------------------------------------------
CPCIEFpgaCSPkt::TxMiscBlockMemoryState
CPCIEFpgaCSPkt::getBlockMemoryState(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  bool runningBit = 0;
  bool idleBit = 0;

  switch(hwChannel)
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENA_RUNNING) ? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENA_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENB_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENB_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENC_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENC_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENBVF1_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENBVF1_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENBVF2_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENBVF2_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENCVF1_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENCVF1_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENCVF2_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENCVF2_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GEND_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GEND_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENE_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENE_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENDVF1_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENDVF1_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENDVF2_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENDVF2_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENEVF1_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENEVF1_IDLE)? true : false);
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    runningBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENEVF2_RUNNING)? true : false);
    idleBit = (getStatusPropVal(portHandle,PCIE_TXSTATUS_REQHEADER_GENEVF2_IDLE)? true : false);
    break;
  default:
    AGT_THROW("CPCIEFpgaCSPkt::getBlockMemoryState: invalid hwchannel val ");
  }

  if(runningBit == 0 && idleBit == 0)
  {
    return FPGATXMISC_BLOCKMEMORY_STATE_STOPPING;
  }

  if(runningBit == 1 && idleBit == 0)
  {
    return FPGATXMISC_BLOCKMEMORY_STATE_RUNNING;
  }

  if(runningBit == 0 && idleBit == 1)
  {
    return FPGATXMISC_BLOCKMEMORY_STATE_STOPPED;
  }

  return FPGATXMISC_BLOCKMEMORY_STATE_INVALID;
}

// -----------------------------------------------------------------------
//UInt32
//CPCIEFpgaCSPkt::getBlockMemoryStates(const AgtPortHandleT& portHandle)
//{
//  UInt32 status = getTxPktStatus(portHandle);
//  UInt32 tmp = (((status >> FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_A_RUNNING_BIT) & 0x3) << 0) |
//                 (((status >> FPGATXMISC_STATUS_REQUEST_HEADER_GENERATOR_B_RUNNING_BIT) & 0x3) << 2);
//                 
//  return tmp;
//}

UInt32 CPCIEFpgaCSPkt::getBlockRepeatFirst( const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& hwChannel )
{
  UInt32 theVal;

  myController->FPGARead( portHandle, mBlockRepeatFirst, theVal );
  UInt32 bitPos = getBitPosRepeatFirst( hwChannel );

  return getBits( theVal, ( UInt8 )bitPos, 8 );
}

void CPCIEFpgaCSPkt::setBlockRepeatFirst( const AgtPortHandleT& portHandle, const EPCIEHwChannelFunction& hwChannel, const UInt32& val )
{
  UInt32 theVal;

  UInt32 bitPos = getBitPosRepeatFirst( hwChannel );
  myController->FPGARead( portHandle, mBlockRepeatFirst, theVal );
  setBits( theVal, ( UInt8 )bitPos, 8, val );
  myController->FPGAWrite( portHandle, mBlockRepeatFirst, theVal );
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getBlockRepeatSecond(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mBlockRepeatSecond,theVal);    
  UInt32 bitPos = getBitPosRepeatSecond(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setBlockRepeatSecond(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatSecond(hwChannel);
  UInt32 theVal ;
  myController->FPGARead(portHandle,mBlockRepeatSecond,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mBlockRepeatSecond,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getBlockRepeatThird(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mBlockRepeatThird,theVal);    
  UInt32 bitPos = getBitPosRepeatThird(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setBlockRepeatThird(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatThird(hwChannel);
  UInt32 theVal ;
  myController->FPGARead(portHandle,mBlockRepeatThird,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mBlockRepeatThird,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getBlockRepeatFourth(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mBlockRepeatFourth,theVal);    
  UInt32 bitPos = getBitPosRepeatFourth(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setBlockRepeatFourth(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatFourth(hwChannel);
  UInt32 theVal ;
  myController->FPGARead(portHandle,mBlockRepeatFourth,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mBlockRepeatFourth,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getBlockRepeatFifth(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mBlockRepeatFifth,theVal);    
  UInt32 bitPos = getBitPosRepeatFifth(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setBlockRepeatFifth(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatFifth(hwChannel);
  UInt32 theVal ;
  myController->FPGARead(portHandle,mBlockRepeatFifth,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mBlockRepeatFifth,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getReqBehRepeatFirst(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mReqBehRepeatFirst,theVal);

  UInt32 bitPos = getBitPosRepeatFirst(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setReqBehRepeatFirst(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatFirst(hwChannel);

  UInt32 theVal;
  myController->FPGARead(portHandle,mReqBehRepeatFirst,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mReqBehRepeatFirst,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getReqBehRepeatSecond(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mReqBehRepeatSecond,theVal);

  UInt32 bitPos = getBitPosRepeatSecond(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setReqBehRepeatSecond(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatSecond(hwChannel);

  UInt32 theVal;
  myController->FPGARead(portHandle,mReqBehRepeatSecond,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mReqBehRepeatSecond,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getReqBehRepeatThird(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mReqBehRepeatThird,theVal);

  UInt32 bitPos = getBitPosRepeatThird(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setReqBehRepeatThird(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatThird(hwChannel);

  UInt32 theVal;
  myController->FPGARead(portHandle,mReqBehRepeatThird,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mReqBehRepeatThird,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getReqBehRepeatFourth(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mReqBehRepeatFourth,theVal);

  UInt32 bitPos = getBitPosRepeatFourth(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setReqBehRepeatFourth(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatFourth(hwChannel);

  UInt32 theVal;
  myController->FPGARead(portHandle,mReqBehRepeatFourth,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mReqBehRepeatFourth,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getReqBehRepeatFifth(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mReqBehRepeatFifth,theVal);

  UInt32 bitPos = getBitPosRepeatFifth(hwChannel);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setReqBehRepeatFifth(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatFifth(hwChannel);

  UInt32 theVal;
  myController->FPGARead(portHandle,mReqBehRepeatFifth,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);
  myController->FPGAWrite(portHandle,mReqBehRepeatFifth,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getCompBehRepeatFirst(const AgtPortHandleT& portHandle,const EPCIECompQueue& queue)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mComplBehRepeatFirst,theVal);

  UInt32 bitPos = getBitPosRepeatFirst(queue);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setCompBehRepeatFirst(const AgtPortHandleT& portHandle,const EPCIECompQueue& queue, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatFirst(queue);

  UInt32 theVal ;
  myController->FPGARead(portHandle,mComplBehRepeatFirst,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);

  myController->FPGAWrite(portHandle,mComplBehRepeatFirst,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getCompBehRepeatSecond(const AgtPortHandleT& portHandle,const EPCIECompQueue& queue)
{
  UInt32 theVal ;
  myController->FPGARead(portHandle,mComplBehRepeatSecond,theVal);

  UInt32 bitPos = getBitPosRepeatFirst(queue);

  return getBits(theVal,(UInt8)bitPos,8);
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setCompBehRepeatSecond(const AgtPortHandleT& portHandle,const EPCIECompQueue& queue, const UInt32& val)
{
  UInt32 bitPos = getBitPosRepeatFirst(queue);

  UInt32 theVal ;
  myController->FPGARead(portHandle,mComplBehRepeatSecond,theVal);

  setBits(theVal,(UInt8)bitPos,8,val);

  myController->FPGAWrite(portHandle,mComplBehRepeatSecond,theVal);
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getBlockPointers(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel,const bool& first)
{
  UInt32 tmp = 0;
  UInt32 bitPos = (first ? PCIEFPGATXMISC_POINTERS_FIRST_ADDR_BITPOS:PCIEFPGATXMISC_POINTERS_LAST_ADDR_BITPOS);
  
  switch(hwChannel)
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    myController->FPGARead(portHandle,mBlockAPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_B:
    myController->FPGARead(portHandle,mBlockBPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_C:
    myController->FPGARead(portHandle,mBlockCPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_D:
     myController->FPGARead(portHandle,mBlockDPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_E:
     myController->FPGARead(portHandle,mBlockEPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    myController->FPGARead(portHandle,mBlockBVF1Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    myController->FPGARead(portHandle,mBlockBVF2Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    myController->FPGARead(portHandle,mBlockCVF1Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    myController->FPGARead(portHandle,mBlockCVF2Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    myController->FPGARead(portHandle,mBlockDVF1Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    myController->FPGARead(portHandle,mBlockDVF2Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    myController->FPGARead(portHandle,mBlockEVF1Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    myController->FPGARead(portHandle,mBlockEVF2Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  default:
    AGT_THROW("CPCIEFpgaCSPkt::getBlockPointers: Invalid prop ");
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setBlockPointers(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel,const bool& first,const UInt32& val)
{
  UInt32 tmp = 0;
  UInt32 bitPos = (first ? PCIEFPGATXMISC_POINTERS_FIRST_ADDR_BITPOS:PCIEFPGATXMISC_POINTERS_LAST_ADDR_BITPOS);
  
  switch(hwChannel)
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    myController->FPGARead(portHandle,mBlockAPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockAPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    myController->FPGARead(portHandle,mBlockBPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockBPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    myController->FPGARead(portHandle,mBlockCPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockCPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    myController->FPGARead(portHandle,mBlockDPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockDPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    myController->FPGARead(portHandle,mBlockEPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockEPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    myController->FPGARead(portHandle,mBlockBVF1Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockBVF1Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    myController->FPGARead(portHandle,mBlockBVF2Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockBVF2Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    myController->FPGARead(portHandle,mBlockCVF1Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockCVF1Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    myController->FPGARead(portHandle,mBlockCVF2Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockCVF2Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    myController->FPGARead(portHandle,mBlockDVF1Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockDVF1Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    myController->FPGARead(portHandle,mBlockDVF2Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockDVF2Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    myController->FPGARead(portHandle,mBlockEVF1Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockEVF1Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    myController->FPGARead(portHandle,mBlockEVF2Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mBlockEVF2Pointers,tmp);
    break;
  default:
    AGT_THROW("CPCIEFpgaCSPkt::setBlockPointers: Invalid prop ");
  }
}

// -----------------------------------------------------------------------
UInt32
CPCIEFpgaCSPkt::getReqBehPointers(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel,const bool& first)
{
  UInt32 tmp = 0;
  UInt32 bitPos = (first ? PCIEFPGATXMISC_POINTERS_FIRST_ADDR_BITPOS:PCIEFPGATXMISC_POINTERS_LAST_ADDR_BITPOS);
  
  switch(hwChannel)
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    myController->FPGARead(portHandle,mReqBehAPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_B:
    myController->FPGARead(portHandle,mReqBehBPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_C:
    myController->FPGARead(portHandle,mReqBehCPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_D:
    myController->FPGARead(portHandle,mReqBehDPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_E:
    myController->FPGARead(portHandle,mReqBehEPointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    myController->FPGARead(portHandle,mReqBehBVF1Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    myController->FPGARead(portHandle,mReqBehBVF2Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    myController->FPGARead(portHandle,mReqBehCVF1Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    myController->FPGARead(portHandle,mReqBehCVF2Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    myController->FPGARead(portHandle,mReqBehDVF1Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    myController->FPGARead(portHandle,mReqBehDVF2Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    myController->FPGARead(portHandle,mReqBehEVF1Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    myController->FPGARead(portHandle,mReqBehEVF2Pointers,tmp);
    return getBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE);
  default:
    AGT_THROW("CPCIEFpgaCSPkt::setReqBehPointers: Invalid prop ");
  }
}

// -----------------------------------------------------------------------
void
CPCIEFpgaCSPkt::setReqBehPointers(const AgtPortHandleT& portHandle,const EPCIEHwChannelFunction& hwChannel,const bool& first,const UInt32& val)
{
  UInt32 tmp = 0;
  UInt32 bitPos = (first ? PCIEFPGATXMISC_POINTERS_FIRST_ADDR_BITPOS:PCIEFPGATXMISC_POINTERS_LAST_ADDR_BITPOS);
  
  switch(hwChannel)
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    myController->FPGARead(portHandle,mReqBehAPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehAPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    myController->FPGARead(portHandle,mReqBehBPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehBPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    myController->FPGARead(portHandle,mReqBehCPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehCPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    myController->FPGARead(portHandle,mReqBehBVF1Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehBVF1Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    myController->FPGARead(portHandle,mReqBehBVF2Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehBVF2Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    myController->FPGARead(portHandle,mReqBehCVF1Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehCVF1Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    myController->FPGARead(portHandle,mReqBehCVF2Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehCVF2Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    myController->FPGARead(portHandle,mReqBehDPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehDPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    myController->FPGARead(portHandle,mReqBehEPointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehEPointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    myController->FPGARead(portHandle,mReqBehDVF1Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehDVF1Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    myController->FPGARead(portHandle,mReqBehDVF2Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehDVF2Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    myController->FPGARead(portHandle,mReqBehEVF1Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehEVF1Pointers,tmp);
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    myController->FPGARead(portHandle,mReqBehEVF2Pointers,tmp);
    setBits(tmp,(UInt8)bitPos,PCIEFPGATXMISC_POINTERS_BITSIZE,val);
    myController->FPGAWrite(portHandle,mReqBehEVF2Pointers,tmp);
    break;
  default:
    AGT_THROW("CPCIEFpgaCSPkt::setReqBehPointers: Invalid prop ");
  }
}

UInt32 CPCIEFpgaCSPkt::getCompBehPointers( const AgtPortHandleT& portHandle, const EPCIECompQueue& queue, const bool& first )
{
  UInt32 tmp = 0;
  UInt32 bitPos = ( first ? PCIEFPGATXMISC_POINTERS_FIRST_ADDR_BITPOS : PCIEFPGATXMISC_POINTERS_LAST_ADDR_BITPOS );

  switch( queue )
  {
  case PCIE_COMPQUEUE_0:
    myController->FPGARead( portHandle, mReqComplQueue0Pointers, tmp );
    return getBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE );

  case PCIE_COMPQUEUE_1:
    myController->FPGARead( portHandle, mReqComplQueue1Pointers, tmp );
    return getBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE );

  case PCIE_COMPQUEUE_2:
    myController->FPGARead( portHandle, mReqComplQueue2Pointers, tmp );
    return getBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE );

  case PCIE_COMPQUEUE_3:
    myController->FPGARead( portHandle, mReqComplQueue3Pointers, tmp );
    return getBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE );

  case PCIE_COMPQUEUE_4:
    myController->FPGARead( portHandle, mReqComplQueue4Pointers, tmp );
    return getBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE );

  default:
    AGT_THROW( "CPCIEFpgaCSPkt::setCompBehPointers: Invalid prop " );
  }
}

void CPCIEFpgaCSPkt::setCompBehPointers( const AgtPortHandleT& portHandle, const EPCIECompQueue& queue, const bool& first, const UInt32& val )
{
  UInt32 tmp = 0;
  UInt32 bitPos = ( first ? PCIEFPGATXMISC_POINTERS_FIRST_ADDR_BITPOS : PCIEFPGATXMISC_POINTERS_LAST_ADDR_BITPOS );

  switch( queue )
  {
  case PCIE_COMPQUEUE_0:
    myController->FPGARead( portHandle, mReqComplQueue0Pointers, tmp );
    setBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE, val );
    myController->FPGAWrite( portHandle, mReqComplQueue0Pointers, tmp );
    break;

  case PCIE_COMPQUEUE_1:
    myController->FPGARead( portHandle, mReqComplQueue1Pointers, tmp );
    setBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE, val );
    myController->FPGAWrite( portHandle, mReqComplQueue1Pointers, tmp );
    break;

  case PCIE_COMPQUEUE_2:
    myController->FPGARead( portHandle, mReqComplQueue2Pointers, tmp );
    setBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE, val );
    myController->FPGAWrite( portHandle, mReqComplQueue2Pointers, tmp );
    break;

  case PCIE_COMPQUEUE_3:
    myController->FPGARead( portHandle, mReqComplQueue3Pointers, tmp );
    setBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE, val );
    myController->FPGAWrite( portHandle, mReqComplQueue3Pointers, tmp );
    break;

  case PCIE_COMPQUEUE_4:
    myController->FPGARead( portHandle, mReqComplQueue4Pointers, tmp );
    setBits( tmp, ( UInt8 )bitPos, PCIEFPGATXMISC_POINTERS_BITSIZE, val );
    myController->FPGAWrite( portHandle, mReqComplQueue4Pointers, tmp );
    break;

  default:
    AGT_THROW( "CPCIEFpgaCSPkt::setCompBehPointers: Invalid prop " );
  }
}

void CPCIEFpgaCSPkt::setBits( UInt32& val, const UInt8& bitPos, const UInt8& bitSize, const UInt32& bitVal )
{
  UInt32 mask;

  if( bitSize == 32 )
  {
    val = bitVal;
  }
  else
  {
    mask = ( ( 0x1 << bitSize ) - 1 );
    val &= ~( mask << bitPos );
    val |= ( ( mask & bitVal ) << bitPos );
  }
}

UInt32 CPCIEFpgaCSPkt::getBits( const UInt32& val, const UInt8& bitPos, const UInt8& bitSize )
{
  UInt32 mask;

  if( bitSize == 32 )
  {
    return val;
  }
  else
  {
    mask = ( ( 0x1 << bitSize ) - 1 );
    return ( val >> bitPos ) & mask;
  }
}

UInt8 CPCIEFpgaCSPkt::getBitPosRepeatFirst( const EPCIECompQueue& queue )
{
  switch( queue )
  {
  case PCIE_COMPQUEUE_0:
  case PCIE_COMPQUEUE_2:
    return 0;
  case PCIE_COMPQUEUE_1:
  case PCIE_COMPQUEUE_3:
    return 8;
  case PCIE_COMPQUEUE_4:
    return 16;
  default:
    AGT_THROW( "CPCIEFpgaCSPkt::getBitPosRepeatFirst: invalid prop val " );
  }
}

UInt8 CPCIEFpgaCSPkt::getBitPosRepeatFirst( const EPCIEHwChannelFunction& hwChannel )
{
 switch( hwChannel )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    return 0;
  case PCIE_HWCHANNEL_FUNCTION_B:
    return 8;
  default:
    AGT_THROW( "CPCIEFpgaCSPkt::getBitPosRepeatFirst: invalid prop val " );
  }
}

UInt8 CPCIEFpgaCSPkt::getBitPosRepeatSecond( const EPCIEHwChannelFunction& hwChannel )
{
 switch( hwChannel )
  {
  // 0,8 are the bit positions for these h/w channels
  case PCIE_HWCHANNEL_FUNCTION_C:
    return 0;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    return 8;
  default:
    AGT_THROW("CPCIEFpgaCSPkt::getBitPosRepeatSecond: invalid prop val ");
  }

 return 0x0;
}

UInt8 CPCIEFpgaCSPkt::getBitPosRepeatThird( const EPCIEHwChannelFunction& hwChannel )
{
 switch( hwChannel )
  {
  // 0,8,16 are the bit positions for these h/w channels,used for requester behavior 
  // memory repeat register and block mem. repeat register
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    return 0;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    return 8;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    return 16;
  default:
    AGT_THROW( "CPCIEFpgaCSPkt::getBitPosRepeatThird: invalid prop val " );
  }

 return 0x0;
}

UInt8 CPCIEFpgaCSPkt::getBitPosRepeatFourth( const EPCIEHwChannelFunction& hwChannel )
{
  switch(hwChannel)
  {
    // 0,8,16,24 are the bit positions for these h/w channels,used for requester behavior 
    // memory repeat register and block mem. repeat register*/
  case PCIE_HWCHANNEL_FUNCTION_D:
    return 0;
  case PCIE_HWCHANNEL_FUNCTION_E:
    return 8;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    return 16;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    return 24;
  default:
    AGT_THROW( "CPCIEFpgaCSPkt::getBitPosRepeatFourth: invalid prop val " );
  }

  return 0x0;
}

UInt8 CPCIEFpgaCSPkt::getBitPosRepeatFifth( const EPCIEHwChannelFunction& hwChannel )
{
 switch( hwChannel )
  {
  // 0,8 are the bit positions for these h/w channels
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    return 0;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    return 8;
  default:
    AGT_THROW( "CPCIEFpgaCSPkt::getBitPosRepeatFifth: invalid prop val " );
  }

 return 0x0;
}

void CPCIEFpgaCSPkt::setValueSendImmediateArray( const AgtPortHandleT& portHandle, UInt32 hwChannel )
{
  for( UInt32 counter = PCIE_HWCHANNEL_FUNCTION_A; counter < PCIE_HWCHANNEL_FUNCTION_END; counter++ )
  {
    mSendImmediateEnabled[portHandle][counter] = 0; // First clear all channels and then enable
  }

  mSendImmediateEnabled[portHandle][hwChannel] = 1; // Set the value of h/w channel
}

EPCIEHwChannelFunction CPCIEFpgaCSPkt::getValueSendImmediateArray( const AgtPortHandleT& portHandle )
{
  for( UInt32 counter = PCIE_HWCHANNEL_FUNCTION_A; counter < PCIE_HWCHANNEL_FUNCTION_END; counter++ )
  {
    if( 1 == mSendImmediateEnabled[portHandle][counter] ) // Check which channel is enabled and return
    {
      return ( EPCIEHwChannelFunction )counter;
    }
  }

  return PCIE_HWCHANNEL_FUNCTION_END;
}

UInt32 CPCIEFpgaCSPkt::swapDW( const UInt32& val )
{
  return ( ( val & 0x000000ff ) << 24 ) |
         ( ( val & 0x0000ff00 ) <<  8 ) |
         ( ( val & 0x00ff0000 ) >>  8 ) |
         ( ( val & 0xff000000 ) >> 24 );
}

/************************************************************************************/
/*********************** Datagenerator **********************************************/
/************************************************************************************/

UInt32 CPCIEFpgaCSPkt::getDelayComplQueue(const AgtPortHandleT& portHandle,UInt32 queue)
{
  // Register is 24 bit wide !
  UInt32 val = 0;

  switch( ( EPCIECompQueue )queue )
  {
    case PCIE_COMPQUEUE_0:
      myController->FPGARead( portHandle, mComplDelayQueue0, val );
      break;
    case PCIE_COMPQUEUE_1:
      myController->FPGARead( portHandle, mComplDelayQueue1, val );
      break;
    case PCIE_COMPQUEUE_2:
      myController->FPGARead( portHandle, mComplDelayQueue2, val );
      break;
    case PCIE_COMPQUEUE_3:
      myController->FPGARead( portHandle, mComplDelayQueue3, val );
      break;
    case PCIE_COMPQUEUE_4:
      myController->FPGARead( portHandle, mComplDelayQueue4, val );
      break;
  }

  return val;
}

void CPCIEFpgaCSPkt::setDelayComplQueue( const AgtPortHandleT& portHandle, UInt32 queue, UInt32 val )
{
  switch( ( EPCIECompQueue )queue )
  {
    case PCIE_COMPQUEUE_0:
      myController->FPGAWrite( portHandle, mComplDelayQueue0, val );
      break;
    case PCIE_COMPQUEUE_1:
      myController->FPGAWrite( portHandle, mComplDelayQueue1, val );
      break;
    case PCIE_COMPQUEUE_2:
      myController->FPGAWrite( portHandle, mComplDelayQueue2, val );
      break;
    case PCIE_COMPQUEUE_3:
      myController->FPGAWrite( portHandle, mComplDelayQueue3, val );
      break;
    case PCIE_COMPQUEUE_4:
      myController->FPGAWrite( portHandle, mComplDelayQueue4, val );
      break;
  }
}